#ifndef _PHYPIB_H_
#define _PHYPIB_H_

/************************************************************************************
*************************************************************************************
* Public Macros and Definitions
*************************************************************************************
************************************************************************************/

//4g MAC sub-1GHz frequency bands
enum {
  g450__470MHz_c   = 0x00, // 450-470   (US FCC Part 90)
  g470__510MHz_c   = 0x01, // 470-510   (China)
  g779__787MHz_c   = 0x02, // 779-787   (China)
  g863__870MHz_c   = 0x03, // 863-870   (Europe)
  g896__901MHz_c   = 0x04, // 896-901   (US FCC Part 90)
  g901__902MHz_c   = 0x05, // 901-902   (US FCC Part 24) 
  g902__928MHz_c   = 0x06, // 902-928   (US) 
  g917__923_5MHz_c = 0x07, // 917-923.5 (Korea)
  g928__960MHz_c   = 0x08, // 928-960 (US, non-contiguous)
  g950__958MHz_c   = 0x09, // 950-958 (Japan)
};

//4g channel page
//enum {
//  gChannelPageSeven_c = 0x07,
//  gChannelPageEight_c = 0x08,
//};

//4g modulation schemes
//enum {
//  gFilteredFSK_c = 0x00,
//  gOFDM_c        = 0x01,
//  gOQPSK         = 0x02,
//};

/************************************************************************************
* 4g PHY modes for Channel Page Seven
* Freq bands:
**450-470   (US FCC Part 90)
--- Bit position 0: 4.8 kbps; filtered FSK; mod index = 1; channel spacing(kHz)   = 12.5
--- Bit position 1: 9.6 kbps; filtered FSK; mod index = 1/3; channel spacing(kHz) = 12.5
**470-510   (China)
--- Bit position 0: 50 kbps; filtered FSK; mod index = 1.0; channel spacing(kHz)   = 200 
--- Bit position 1: 100 kbps; filtered FSK; mod index = 1.0; channel spacing(kHz)  = 400
--- Bit position 2: 200 kbps; filtered 4FSK; mod index = 1/3; channel spacing(kHz) = 400
**779-787   (China)
**863-870   (Europe)
--- Bit position 0: 50 kbps; filtered FSK; mod index = 1.0; channel spacing(kHz)   = 200
--- Bit position 1: 100 kbps; filtered FSK; mod index = 1.0; channel spacing(kHz)  = 400
--- Bit position 2: 200 kbps; filtered 4FSK; mod index = 1/3; channel spacing(kHz) = 400
**896-901   (US FCC Part 90)
--- Bit position 0: 10 kbps; filtered FSK; mod index = 0.5; channel spacing(kHz) = 25
--- Bit position 1: 20 kbps; filtered FSK; mod index = 0.5; channel spacing(kHz) = 25
--- Bit position 2: 40 kbps; filtered FSK; mod index = 0.5; channel spacing(kHz) = 25
**901-902   (US FCC Part 24)
--- Bit position 0: 10 kbps; filtered FSK; mod index = 0.5; channel spacing(kHz)= 25
--- Bit position 1: 20 kbps; filtered FSK; mod index = 0.5; channel spacing(kHz)= 25
--- Bit position 2: 40 kbps; filtered FSK; mod index = 0.5; channel spacing(kHz)= 25
**902-928   (US) 
--- Bit position 0: 50 kbps; filtered FSK; mod index = 1.0; channel spacing given(kHz) = 200
--- Bit position 1: 150 kbps; filtered FSK; mod index = 0.5; channel spacing given(kHz) = 400
--- Bit position 2: 200 kbps; filtered FSK; mod index = 0.5; channel spacing given(kHz) = 400
**917-923.5 (Korea)
--- Bit position 0: 50 kbps; filtered FSK; mod index = 1.0; channel spacing given in Table 75a
--- Bit position 1: 150 kbps; filtered FSK; mod index = 0.5; channel spacing given in Table 75a
--- Bit position 2: 200 kbps; filtered FSK; mod index = 0.5; channel spacing given in Table 75a
**928-960 (US, non-contiguous)
--- Bit position 0: 10 kbps; filtered FSK; mod index = 0.5; channel spacing given(kHz) = 25
--- Bit position 1: 20 kbps; filtered FSK; mod index = 0.5; channel spacing given(kHz) = 25
--- Bit position 2: 40 kbps; filtered FSK; mod index = 0.5; channel spacing given(kHz) = 25
**950-958 (Japan)
--- Bit position 0: 50 kbps; filtered FSK; mod index = 1.0; channel spacing(kHz) = 200, 400
--- Bit position 1: 100 kbps; filtered FSK; mod index = 1.0; channel spacing(kHz) = 400
--- Bit position 2: 200 kbps; filtered FSK; mod index = 1.0; channel spacing(kHz) = 600
--- Bit position 3: 400 kbps; filtered 4FSK; mod index = 0.33; channel spacing(kHz) = 600
************************************************************************************/

enum{
 gPhyMode0_c = 1<<0, 
 gPhyMode1_c = 1<<1, 
 gPhyMode2_c = 1<<2,
 gPhyMode3_c = 1<<3, 
};

/******************************************************************
** gPhyPibPhyModeSupported_c bitmap defintion                     |
** Bit pos:|    15-9      |     8-4        |          3-0         |
**         |  Reserved    | Frequency band |  Bitmap for PHY mode |
*******************************************************************/

#define gSUNPage_PhyMode_BitOffset_c          0
#define gSUNPage_ModulationScheme_BitOffset_c 20
#define gSUNPage_FreqBand_BitOffset_c         22
#define gSUNPage_ChannelPage_BitOffset_c      27

#define gPhyPibMaxPmbLen_c  4000
#define gPhyPibMinPmbLen_c  4

/* Extended PHY definitions */

#define mHighestGainMinus6_c		( 190 )          // -95dBm
#define mHighestGainMinus12_c		( 176 )			 // -88dBm		
#define mHighestGainMinus24_c		( 154 )			 // -77dBm
#define mHighestGainMinus36_c		( 136 )			 // -68dBm
#define mHighestGainMinus48_c		( 114 )			 // -57dBm

//
enum 
{
  gPhyPibBusyTransceiver_c,
  gPhyPibSuccess_c,
  gPhyPibReadOnly_c,
  gPhyPibInvalidParameter_c,
  gPhyPibUnsupportedAttribute
};

// PHY PIB attributes
enum {
  gPhyPibPhyModeSupported_c = 0x00,
  gPhyPibCurrentMode_c,
  gPhyPibCurrentChannel_c,
  gPhyPibTransmitPower_c,
  gPhyPibFSKPreambleRepetitions_c,
  gPhyPibFSKScramblePSDU_c
};

/************************************************************************************
*************************************************************************************
* Public type definitions
*************************************************************************************
************************************************************************************/

typedef struct phyRFConstatnts_tag  
{
  uint32_t firstChannelFrequency;
  uint32_t channelSpacing;
  uint16_t totalNumChannels;
  uint16_t bitRateReg;
  uint16_t fdevReg;
  uint16_t numRssiMeasurements;
  uint8_t  rxBwReg;
  uint8_t  rxBwAfcReg;
  uint8_t  modulationParam;
  uint8_t  ccaThreshold;
} phyRFConstants_t;

/******************************************************************************
*@CMA
*IMPORTANT NOTE: Accessing limitation of Cortex M0+
*(Need to align this struct to 32 bit addresses) (Address %4 -modulo 4- must equal zero)
******************************************************************************/

#if defined(IAR)
  typedef struct phyPib_tag                                                     
  {

    uint32_t mPIBphyModeSupported;//read-only PIB                               //@CMA, Member redefied to 32 bits                 
    uint32_t mPIBphyCurrentMode;                                                //@CMA, Member redefied to 32 bits
    uint32_t  mPIBphyCurrentChannel;                                            //@CMA, Member redefied to 32 bits
    uint32_t  mPIBphyTransmitPower;                                             //@CMA, Member redefied to 32 bits
    uint32_t mPIBphyFSKPreambleRepetitions;                                     //@CMA, Member redefied to 32 bits
    bool_t   mPIBphyFSKScramblePSDU;
    phyRFConstants_t *pPIBphyRfConstants; //internal used
  } phyPib_t;


  typedef union
  {
    uint32_t shortAccess;
    struct
    {
     unsigned int  phyMode  :4;           //@CMA, LSB
     unsigned int  freqBand :5;
     unsigned int  reserved :7;           //@CMA, MSB
    }bitAccess;
  } phyMode_t;
#endif

#if defined(CW)                                                                 //@CMA The definitions inside this define are the original ones
  typedef struct phyPib_tag                                                     //@CMA, TODO!! For CodeWarrior, Need to align each member of the structure to 4 bytes (limitation of Cortex M0+)
  {

    uint16_t mPIBphyModeSupported;//read-only PIB                        
    uint16_t mPIBphyCurrentMode;
    uint8_t  mPIBphyCurrentChannel;
    uint8_t  mPIBphyTransmitPower;
    uint16_t mPIBphyFSKPreambleRepetitions;
    bool_t   mPIBphyFSKScramblePSDU;
    phyRFConstants_t *pPIBphyRfConstants; //internal used
  } phyPib_t;


  typedef union
  {
    uint16_t shortAccess;
    struct
    {
     unsigned int  phyMode  :4;           //@CMA, LSB
     unsigned int  freqBand :5;
     unsigned int  reserved :7;           //@CMA, MSB
    }bitAccess;
  } phyMode_t;
#endif
/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/

extern phyPib_t gPhyPib;
extern phyRFConstants_t const phyPibRFConstants[];

/************************************************************************************
*************************************************************************************
* Public prototypes
*************************************************************************************
************************************************************************************/

void PhyPib_InitOnStartUp(void);
uint8_t PhyPib_SetCurrentPhyMode(uint16_t phyMode);
uint8_t PhyPib_SetCurrentChannel(uint8_t channel);
uint8_t PhyPib_SetCurrentTransmitPower(uint8_t pwrLevel);
uint8_t PhyPib_SetCurrentFSKPreambleRepetitions(uint16_t preambleLength);
uint8_t PhyPib_SetCurrentFSCScrablePSDU(bool_t scrablePSDU);
uint8_t PhyPib_Get(uint8_t attribute, void * pAttributeValue);
uint8_t PhyPib_Set(uint8_t attribute, void * pAttributeValue);


/************************************************************************************
*************************************************************************************
* Private prototypes
*************************************************************************************
************************************************************************************/
void PhyPib_RFUpdateDCfreeEncoding(void);
void PhyPib_RFUpdatePreambleLength(void);
void PhyPib_RFUpdatePowerStep(void);            
void PhyPib_RFUpdateRFfrequency(void);
void PhyPib_RFUpdateModulationParameters(void);


#endif /* _PHYPIB_H_ */
